<?php

namespace App\Http\Controllers;

use App\Models\Assignment;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class AssignmentController extends Controller
{
    public function index(): JsonResponse
    {
        $assignments = Assignment::with(['lesson', 'user'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $assignments,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $assignment = Assignment::with(['lesson', 'user'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $assignment,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Lesson_id' => 'required|exists:lessons,Lesson_id',
            'User_id' => 'required|exists:users,User_id',
            'Title' => 'required|string|max:255',
            'Description' => 'required|string',
            'Submitted_at' => 'nullable|date',
            'Status' => 'sometimes|in:pending,submitted,graded',
            'Grade' => 'nullable|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $assignment = Assignment::create($request->only([
            'Lesson_id',
            'User_id',
            'Title',
            'Description',
            'Submitted_at',
            'Status',
            'Grade',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $assignment,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $assignment = Assignment::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Lesson_id' => 'sometimes|exists:lessons,Lesson_id',
            'User_id' => 'sometimes|exists:users,User_id',
            'Title' => 'sometimes|string|max:255',
            'Description' => 'sometimes|string',
            'Submitted_at' => 'nullable|date',
            'Status' => 'sometimes|in:pending,submitted,graded',
            'Grade' => 'nullable|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $assignment->update($request->only([
            'Lesson_id',
            'User_id',
            'Title',
            'Description',
            'Submitted_at',
            'Status',
            'Grade',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $assignment,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $assignment = Assignment::findOrFail($id);
        $assignment->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Assignment deleted successfully',
        ], 200);
    }

    // Additional method to submit an assignment
    public function submitAssignment(Request $request, $id): JsonResponse
    {
        $assignment = Assignment::findOrFail($id);

        if ($assignment->Status !== 'pending') {
            return response()->json([
                'status' => 'error',
                'message' => 'Assignment is not in pending state',
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'Submitted_at' => 'required|date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $assignment->update([
            'Submitted_at' => $request->Submitted_at,
            'Status' => 'submitted',
        ]);

        return response()->json([
            'status' => 'success',
            'data' => $assignment,
            'message' => 'Assignment submitted successfully.',
        ], 200);
    }

    // Additional method to grade an assignment
    public function gradeAssignment(Request $request, $id): JsonResponse
    {
        $assignment = Assignment::findOrFail($id);

        if ($assignment->Status !== 'submitted') {
            return response()->json([
                'status' => 'error',
                'message' => 'Assignment is not submitted yet',
            ], 400);
        }

        $validator = Validator::make($request->all(), [
            'Grade' => 'required|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $assignment->update([
            'Grade' => $request->Grade,
            'Status' => 'graded',
        ]);

        return response()->json([
            'status' => 'success',
            'data' => $assignment,
            'message' => 'Assignment graded successfully.',
        ], 200);
    }
}
